/*
 * Copyright (c) 2018-2019 Texas Instruments Incorporated - http://www.ti.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== NAT.syscfg.js ========
 *  NAT configuration support
 */

"use strict";
/* global exports, system */

//console.log("load: ti/ndk/NAT");

let util = system.getScript("/ti/ndk/util.js");

/*
 *  ======== modules ========
 *  Express dependencies for other modules
 *
 *  Invoked on any configuration change to the given instance.
 */
function modules(inst)
{
    let modules = new Array();

    modules.push({
        name: "general",
        moduleName: "/ti/ndk/General"
    });

    return (modules);
}

/*
 *  ======== validate_instance ========
 *  Validate given instance and report conflicts
 *
 *  This function is not allowed to modify the instance state.
 */
function validate_instance(inst, vo, getRef)
{
    /* Server interface ID */
    if(inst.ifIdx < 1 ) {
        vo["ifIdx"].errors.push("Must be greater than 0");
    }

    /* virtualIpAddr */
    if (!util.validateIPAddr(inst.virtualIpAddr)) {
        vo["virtualIpAddr"].errors.push("Invalid IP address");
    }

    /* virtualMask */
    if (!util.validateIPAddr(inst.virtualMask)) {
        vo["virtualMask"].errors.push("Invalid IP address");
    }
}

/*
 *  ======== longDescription ========
 *  Intro splash on GUI
 */
let longDescription =
    "Create and configure NAT server instances. Multiple NAT server " +
    "instances are supported, but each one must be configured to have " +
    "a unique port number.";

/*
 *  ======== config_module ========
 *  Define the config params of an instance
 */
let config_instance = [
    {
        displayName: "NAT Server Flags",
        config: [
            {
                name: "IfIdxValid",
                displayName: "Interface index is valid",
                default: true,
                hidden: true,
                longDescription: `
CIS_FLG_IFIDXVALID - Specifies if the IfIdx field is valid.

[More ...](/ndk/ConfigDoc.html#ti_ndk_NAT_IfIdxValid)`,
                documentation: `
This setting is part of the common argument structure for NDK services. It alters
the Mode field documented in the
[NDK API Guide](NDK_API_Reference.html#common-argument-structure).
`
            },
            {
                name: "ResolveIP",
                displayName: "Resolve interface to IP address",
                default: false,
                hidden: true,
                longDescription: `
CIS_FLG_RESOLVEIP - Requests that IfIdx be resolved to an IP address before
service execution is initiated.

[More ...](/ndk/ConfigDoc.html#ti_ndk_NAT_ResolveIP)`,
                documentation: `
This setting is part of the common argument structure for NDK services. It alters
the Mode field documented in the
[NDK API Guide](NDK_API_Reference.html#common-argument-structure).
`
            },
            {
                name: "CallByIP",
                displayName: "Call using IP",
                default: false,
                hidden: true,
                longDescription: `
CIS_FLG_CALLBYIP - Specifies that the service should be invoked by IP address

[More ...](/ndk/ConfigDoc.html#ti_ndk_NAT_CallByIP)`,
        documentation: `
This setting is part of the common argument structure for NDK services. It alters
the Mode field documented in the
[NDK API Guide](NDK_API_Reference.html#common-argument-structure).
`
            },
            {
                name: "RestartIPTerm",
                displayName: "Restart service on IPTERM",
                default: false,
                hidden: true,
                longDescription: `
CIS_FLG_RESTARTIPTERM - A service that is dependent on a valid IP address.

[More ...](/ndk/ConfigDoc.html#ti_ndk_NAT_RestartIPTerm)`,
                documentation: `
This setting is part of the common argument structure for NDK services. It alters
the Mode field documented in the
[NDK API Guide](NDK_API_Reference.html#common-argument-structure).
`
            }
        ]
    },
    /*
     * The next menu item has to do with configuring this service for
     * devices with multiple interfaces. NDK syscfg is targeting E4 devices
     * right now, so these menu options are left hidden until we begin to support
     * multiple interface NDK targets.
     */
    {
        name: "ifIdx",
        displayName: "Server interface ID",
        hidden: true,
        default: 1,
        description: "The physical device index on which the Telnet server " +
            "shall be executed. Must be greater than zero."
    },
    {
        name: "virtualIpAddr",
        displayName: "Virtual IP Address",
        default: "0.0.0.0",
        readOnly: false,
        description: "Specifies the NAT group virtual network IP address.",
        longDescription: `
Sets the NAT Group virtual network address

[More ...](/ndk/ConfigDoc.html#ti_ndk_NAT_virtualIpAddr)`,
        documentation: `
More information on the NAT Server Service can be found in the
[NDK API Guide](NDK_API_Reference.html#network-address-translation-nat-service).
`
    },
    {
        name: "virtualMask",
        displayName: "Virtual IP Mask",
        default: "255.255.254.0",
        readOnly: false,
        description: "Specifies the subnet mask of the NAT group virtual network.",
        longDescription: `
Sets the subnet mask of NAT Group virtual network

[More ...](/ndk/ConfigDoc.html#ti_ndk_NAT_virtualMask)`,
        documentation: `
More information on the NAT Server Service can be found in the
[NDK API Guide](NDK_API_Reference.html#network-address-translation-nat-service).
`
    },
    {
        name: "mtu",
        displayName: "MTU",
        default: "1500",
        readOnly: false,
        description: "Specifies the IP MTU limit.",
        longDescription: `
Sets the IP MTU Limit (1500 for Ethernet, 1492 for PPPoE, etc.)

[More ...](/ndk/ConfigDoc.html#ti_ndk_NAT_mtu)`,
        documentation: `
More information on the NAT Server Service can be found in the
[NDK API Guide](NDK_API_Reference.html#network-address-translation-nat-service).
`
    },
    {
        name: "pCbSrv",
        displayName: "Service report function",
        default: "serviceReport",
        description: "NAT service reporting function.",
        longDescription: `
Service report function used by the NAT server. The default value will use the
report function generated by SysConfig.

[More ...](/ndk/ConfigDoc.html#ti_ndk_NAT_pCbSrv)`,
        documentation: `
Information on adding your own service report function can be found in the
[NDK User's Guide](NDK_Users_Guide.html#adding-status-report-services).
`
    }
];

/*
 *  ======== tfxn ========
 *  Template helper functions
 */
let tfxn = {
    cisargs_mode: function(inst)
    {
        let flags = [];

        if (inst.IfIdxValid) flags.push("CIS_FLG_IFIDXVALID");
        if (inst.ResolveIP) flags.push("CIS_FLG_RESOLVEIP");
        if (inst.CallByIP) flags.push("CIS_FLG_CALLBYIP");
        if (inst.RestartIPTerm) flags.push("CIS_FLG_RESTARTIPTERM");

        if (flags.length == 0) {
            return ("0");
        }
        else {
            return (flags.join(" | "));
        }
    }
};

/*
 *  ======== base ========
 *  Module definition object
 */
let base = {
    displayName: "NAT Server",
    description: "NDK NAT Server configuration",
    longDescription: longDescription,
    config: config_instance,
    validate: validate_instance,
    moduleStatic: {
        modules: modules
    },

    templates: {
        "/ti/ndk/Config.c.xdt": "/ti/ndk/NAT.Config.c.xdt"
    },
    tfxn: tfxn
};

/* export the module */
exports = base;
