/*
 * Copyright (c) 2018-2019 Texas Instruments Incorporated - http://www.ti.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== DHCP.syscfg.js ========
 *  DHCP configuration support
 */

"use strict";
/* global exports, system */

//console.log("load: ti/ndk/DHCP");

let util = system.getScript("/ti/ndk/util.js");

/*
 *  ======== modules ========
 *  Express dependencies for other modules
 *
 *  Invoked on any configuration change to the given instance.
 */
function modules(inst)
{
    let modules = new Array();

    modules.push({
        name: "general",
        moduleName: "/ti/ndk/General"
    });

    return (modules);
}

/*
 *  ======== validate_instance ========
 *  Validate given instance and report conflicts
 *
 *  This function is not allowed to modify the instance state.
 */
function validate_instance(inst, vo, getRef)
{
    /* Server interface ID */
    if(inst.ifIdx < 1 ) {
        vo["ifIdx"].errors.push("Must be greater than 0");
    }

    /* ipAddrPoolBase */
    if (!util.validateIPAddr(inst.ipAddrPoolBase)) {
        vo["ipAddrPoolBase"].errors.push("Invalid IP address");
    }
}

/*
 *  ======== longDescription ========
 *  Intro splash on GUI
 */
let longDescription =
    "Create and configure DHCP server instances. Multiple DHCP server " +
    "instances are supported, but each one must be configured to have " +
    "a unique port number.";

/*
 *  ======== config_module ========
 *  Define the config params of an instance
 */
let config_instance = [
    {
        displayName: "DHCP Server Flags",
        config: [
            {
                name: "localDomain",
                displayName: "Report the local domain name assigned to the " +
                    "virtual network to clients (DHCPS_FLG_LOCALDOMAIN)",
                default: false,
                description: "Report the local domain name assigned to the " +
                    "virtual network to clients",
                longDescription: `
Causes DHCPS to report the local domain name assigned to the virtual network to
clients. If this flag is not set, DHCPS reports the public domain name to
clients

[More ...](/ndk/ConfigDoc.html#ti_ndk_DHCP_localDomain)`,
                documentation: `
More information on the DHCP Server flags can be found in the
[NDK API Flag](NDK_API_Reference.html#dhcp-server-parameter-structure).
`
            },
            {
                name: "localDNS",
                displayName: "Report ourselves as the local DNS server to " +
                    "clients (DHCPS_FLG_LOCALDNS)",
                default: false,
                description: "Report ourselves as the local DNS server to " +
                    "clients",
                longDescription: `
Causes DHCPS to report its own IP address as the local DNS server to clients.
If this flag is not set, DHCPS reports the DNS servers as contained in the
SYSINFO portion of the configuration.

[More ...](/ndk/ConfigDoc.html#ti_ndk_DHCP_localDNS)`,
                documentation: `
More information on the DHCP Server flags can be found in the
[NDK API Flag](NDK_API_Reference.html#dhcp-server-parameter-structure).
`
            }
        ]
    },
    /*
     * The next menu item has to do with configuring this service for
     * devices with multiple interfaces. NDK syscfg is targeting E4 devices
     * right now, so these menu options are left hidden until we begin to support
     * multiple interface NDK targets.
     */
    {
        name: "ifIdx",
        displayName: "Server interface ID",
        hidden: true,
        default: 1,
        description: "The physical device index on which the Telnet server " +
            "shall be executed. Must be greater than zero."
    },
    {
        name: "ipAddrPoolBase",
        displayName: "IP Address Pool Base",
        default: "192.168.1.2",
        readOnly: false,
        description: "The first IP address of the DHCP server address pool.",
        longDescription: `
The first IP address (in dotted decimal notation) of the address pool.

[More ...](/ndk/ConfigDoc.html#ti_ndk_DHCP_ipAddrPoolBase)`,
        documentation: `
More information on the DHCP Server Service can be found in the
[NDK API Flag](NDK_API_Reference.html#dhcp-server-service).
`
    },
    {
        name: "ipAddrPoolCount",
        displayName: "IP Address Pool Count",
        default: "253",
        readOnly: false,
        description: "The number of IP addresses in the DHCP server address pool.",
        longDescription: `
The number of addresses in the address pool.

[More ...](/ndk/ConfigDoc.html#ti_ndk_DHCP_ipAddrPoolCount)`,
        documentation: `
More information on the DHCP Server Service can be found in the
[NDK API Flag](NDK_API_Reference.html#dhcp-server-service).
`
    },
    {
        name: "pCbSrv",
        displayName: "Service report function",
        default: "serviceReport",
        description: "DHCP service reporting function.",
        longDescription: `
Service report function used by the DHCP Server. The default value will use the
report function generated by SysConfig.

[More ...](/ndk/ConfigDoc.html#ti_ndk_DHCP_pCbSrv)`,
        documentation: `
Information on adding your own service report function can be found in the
[NDK User's Guide](NDK_Users_Guide.html#adding-status-report-services).
`
    }
];

/*
 *  ======== tfxn ========
 *  Template helper functions
 */
let tfxn = {
    cisargs_mode: function(inst)
    {
        let flags = [];

        if (inst.localDomain) flags.push("DHCPS_FLG_LOCALDOMAIN");
        if (inst.localDNS) flags.push("DHCPS_FLG_LOCALDNS");

        if (flags.length == 0) {
            return ("0");
        }
        else {
            return (flags.join(" | "));
        }
    }
};

/*
 *  ======== base ========
 *  Module definition object
 */
let base = {
    displayName: "DHCP Server",
    description: "NDK DHCP Server configuration",
    longDescription: longDescription,
    config: config_instance,
    validate: validate_instance,
    moduleStatic: {
        modules: modules
    },

    templates: {
        "/ti/ndk/Config.c.xdt": "/ti/ndk/DHCP.Config.c.xdt"
    },
    tfxn: tfxn
};

/* export the module */
exports = base;
